function CDX = update_model_price_index(CDX, discounts_IMM, maturities, start_date_num, end_date_num)
% --------------------------------------------------------------------------------------------------
% Update the model implied coupon rate for an index.
% --------------------------------------------------------------------------------------------------
% CDX                           ... credit index structure (see 'all_steps_in_a_row.m')
% discounts_IMM                 ... structure with discount curves matching IMM dates
% maturities                    ... which maturities to update (indicator vector), default: update all
% --------------------------------------------------------------------------------------------------
% sample call: update_model_price_index(cdx_test, discounts_IMM, [1 0 0], cdx_test.dates{1}(1), cdx_test.dates{1}(1))
%              update_model_price_index(cdx_mor, discounts_IMM, [1 0 0], datenum('12/05/2005'),
%              datenum('12/05/2005'))
% --------------------------------------------------------------------------------------------------

% Default is to update model-implied index price for all maturities
if (nargin == 2)
   maturities = ones(length(CDX.dates), 1);
end

for i=1:length(maturities)
    if (maturities(i) == 1)
        % Determine relevant date range
        start_pos = find(CDX.dates{i} >= start_date_num, 1, 'first');
        end_pos = find(CDX.dates{i} <= end_date_num, 1, 'last');
        used_range = start_pos:end_pos;
        used_dates = CDX.dates{i}(used_range);
        if (i==1)
            used_range_5yr = used_range;
            used_dates_5yr = used_dates;
        end
        if (isempty(used_dates))
            continue;
        end

        % Determine maximum needed horizon
        tmp = [5 7 10];
        T_max = max(tmp(logical(maturities))) + 0.25;
        used_horizons = CDX.T{i}(used_range);
    
        % Extract relevant discount curves
        [discounts, discounts_inbetween] = get_discounts(used_dates, discounts_IMM, T_max);

        % Determine used range for index members
        [trash, pos_5yr] = intersect_sorted(used_dates_5yr, used_dates);
        used_range_index_member = used_range_5yr(pos_5yr);
        
        % Solve for model implied coupon rate
        npv_fixed_leg = value_fixed_leg_index(used_dates, used_horizons, CDX.index_market_price{i}(used_range), CDX.portfolio, ...
                                              CDX.index_members(used_range_index_member,:), discounts, discounts_inbetween, maturities, ...
                                              CDX.AJD_common_factor, CDX.y0);
        npv_default_leg = value_default_leg_index(used_dates, used_horizons, CDX.index_market_price{i}(used_range), CDX.portfolio, ...
                                                  CDX.index_members(used_range_index_member,:), discounts, discounts_inbetween, maturities, ...
                                                  CDX.LGD.mean, CDX.AJD_common_factor, CDX.y0);
        % If coupon is equal to zero, use 1bps
        positive = (CDX.index_market_price{i}(used_range) > 0);
        c = zeros(length(positive), 1);
        c(positive,:) = npv_default_leg(positive,:) ./ npv_fixed_leg(positive,:) .* CDX.index_market_price{i}(used_range(positive),:);
        c(~positive,:) = npv_default_leg(~positive,:) ./ npv_fixed_leg(~positive,:);
                
        % Create fields in CDX structure (if not already exists) for model-implied prices
        if (~isfield(CDX, 'index_model_price'))
            CDX.index_model_price{i} = [];
        end
        if (~isfield(CDX, 'index_PV01'))
            CDX.index_PV01{i} = [];
        end

        % Expand size of field for model-implied prices (if necessary)
        if (length(CDX.index_model_price) < i)
            CDX.tranche_model_upfront{i} = zeros(length(CDX.dates{i}), 1);
        end
        if (length(CDX.index_PV01) < i)
            CDX.index_PV01{i} = zeros(length(CDX.dates{i}), 1);
        end
            
        % Update index with model-implied price
        CDX.index_model_price{i}(used_range,:) = c;
        
        % Update PV01 for index
        CDX.index_PV01{i}(used_range(positive),:) = npv_fixed_leg(positive,:) ./ CDX.index_market_price{i}(used_range(positive),:) * 1e4;
        CDX.index_PV01{i}(used_range(~positive),:) = npv_fixed_leg(~positive,:) * 1e4;
    end
end

% Calculate RMSE_index
if (~isfield(CDX, 'rmse_index'))
    CDX.rmse_index = zeros(size(CDX.dates{1}));
end
used_dates = CDX.dates{1}(used_range_index_member);
errors = zeros(length(used_dates), 1);
num_errors = zeros(length(used_dates), 1);
for i=1:length(maturities)
    if (maturities(i) == 1)
        % Determine matching date range
        [trash, pos_error, pos_cdx] = intersect_sorted(used_dates, CDX.dates{i});
        if (isempty(pos_error))
            continue;
        end
        
        % Extract market and model price
        market_price = CDX.index_market_price{i}(pos_cdx);
        model_price = CDX.index_model_price{i}(pos_cdx);
        
        % Calculate error (use zero, if market_price == 0)
        errors(pos_error) = errors(pos_error) + ((market_price - model_price) ./ max(market_price,1e-10)).^2 .* (market_price > 0);
        num_errors(pos_error) = num_errors(pos_error) + 1;
    end
end
CDX.rmse_index(used_range_index_member) = sqrt(errors ./ max(1,num_errors));













